#!/bin/bash
set -e

# Define variables
ISO_FILENAME="SMTX_VMTOOLS-3.2.2-2501210639.iso"
ISO_FILE="/tmp/${ISO_FILENAME}"
MOUNT_POINT="/mnt/wuke-vmtools"

# Extract version from ISO filename
VMTOOLS_VERSION=$(echo "$ISO_FILENAME" | awk -F'-' '{print $2}')
echo "Extracted VMTools version: $VMTOOLS_VERSION"

# Detect gateway and set ISO_URL
GATEWAY=$(ip route | grep default | awk '{print $3}')
echo "Detected gateway: $GATEWAY"

if [[ $GATEWAY == 172.29.16.* ]]; then
  ISO_URL="http://172.29.16.5/VMTOOLS/${VMTOOLS_VERSION}/${ISO_FILENAME}"
elif [[ $GATEWAY == 172.29.48.* ]]; then
  ISO_URL="http://172.29.48.5/VMTOOLS/${VMTOOLS_VERSION}/${ISO_FILENAME}"
elif [[ $GATEWAY == 172.29.80.* ]]; then
  ISO_URL="http://172.29.80.5/VMTOOLS/${VMTOOLS_VERSION}/${ISO_FILENAME}"
elif [[ $GATEWAY == 172.29.112.* ]]; then
  ISO_URL="http://172.29.112.5/VMTOOLS/${VMTOOLS_VERSION}/${ISO_FILENAME}"
else
  echo "Unsupported gateway network: $GATEWAY"
  exit 1
fi

echo "Selected ISO URL: $ISO_URL"

# Function to check and install a package if missing
install_package_if_missing() {
  local package_name=$1

  if command -v yum >/dev/null 2>&1; then
    if ! rpm -q "$package_name" >/dev/null 2>&1; then
      echo "Installing $package_name using yum..."
      yum install -y "$package_name"
    else
      echo "Package $package_name already installed (yum)."
    fi
  elif command -v apt >/dev/null 2>&1; then
    if ! dpkg -s "$package_name" >/dev/null 2>&1; then
      echo "Installing $package_name using apt..."
      apt update
      apt install -y "$package_name"
    else
      echo "Package $package_name already installed (apt)."
    fi
  else
    echo "Unsupported OS: yum or apt not found."
    exit 1
  fi
}

# 1. Install required packages
install_package_if_missing tar
install_package_if_missing bzip2
install_package_if_missing curl

# 2. Check if ISO URL is accessible
echo "Checking if ISO URL is accessible..."
if ! curl --head --silent --fail "$ISO_URL" >/dev/null; then
  echo "Error: ISO URL $ISO_URL is not accessible."
  exit 1
fi

# 3. Download the ISO file
echo "Downloading VMTools ISO..."
curl -o "$ISO_FILE" "$ISO_URL"

# 4. Check if /mnt/wuke-vmtools exists
if [ -d "$MOUNT_POINT" ]; then
  echo "Directory $MOUNT_POINT exists, removing..."
  umount "$MOUNT_POINT" || true
  rm -rf "$MOUNT_POINT"
fi

echo "Creating directory $MOUNT_POINT..."
mkdir -p "$MOUNT_POINT"

# 5. Mount the ISO
echo "Mounting ISO file to $MOUNT_POINT..."
mount -o loop "$ISO_FILE" "$MOUNT_POINT"

# 6. Execute the installation script
echo "Running VMTools installation script..."
bash "$MOUNT_POINT/SMTX_VM_TOOLS_INSTALL.sh"

# 7. Cleanup
echo "Cleaning up temporary files..."
umount "$MOUNT_POINT" || true
rm -f "$ISO_FILE"
rm -rf "$MOUNT_POINT"

echo "Installation and cleanup completed."